function bacteria_growth_examples
% Bacterial Growth Models Explorer
% Interactively see how K (carrying capacity), r (growth rate),
% and lambda (lag time) affect Exponential, Logistic, and Modified Gompertz growth.
%
% Exponential (no lag):
%   N(t) = N0 * exp(r*t)
%
% Logistic (no lag):
%   N(t) = K / (1 + ((K - N0)/N0) * exp(-r*t))
%
% Modified Gompertz (Zwietering et al.) with lag:
%   A = ln(K/N0)
%   N(t) = N0 * exp( A * exp( -exp( (r*exp(1)/A) * (λ - t) + 1 ) ) )
%
% Notes:
%   - Exponential & Logistic have no lag; Gompertz uses λ.
%   - "Y-axis scale" control: Log (fixed) vs Linear (auto).
%   - Figure & axes background set to #ececec.

    %% Defaults and time grid
    T_max = 48;                  % hours
    dt    = 0.05;
    t     = 0:dt:T_max;

    defaults.N0  = 1e4;          % initial population (arbitrary units)
    defaults.K   = 1e10;         % carrying capacity
    defaults.r   = 0.6;          % 1/h
    defaults.lam = 3.0;          % hours (lag time, Gompertz only)

    %% Colors / fonts
    bgGrey   = [236 236 236] / 255;    % #ececec
    kColor   = [0.18 0.40 0.85];       % blue-ish for K reference
    lagColor = [0.90 0.40 0.10];       % orange-ish for λ
    shadeCol = lagColor;               % shaded lag region color

    % Font (fallback if Nunito is unavailable)
    appFont = 'Nunito';
    if ~localFontAvailable(appFont), appFont = get(groot,'defaultUicontrolFontName'); end

    %% Figure
    f  = figure('Name','Bacterial Growth Models Explorer',...
                'NumberTitle','off','Color',bgGrey,'Position',[100 100 1100 620]);

    % App-wide font defaults
    set(f, 'DefaultAxesFontName', appFont, ...
           'DefaultTextFontName', appFont, ...
           'DefaultLegendFontName', appFont, ...
           'DefaultUicontrolFontName', appFont);

    %% Axes
    ax = axes('Parent',f,'Position',[0.08 0.18 0.62 0.75]);
    hold(ax,'on'); grid(ax,'on');

    % Visibility upgrades
    ax.FontSize = 16;
    ax.LabelFontSizeMultiplier = 1.15;
    ax.TitleFontSizeMultiplier = 1.1;
    ax.LineWidth = 1.25;

    % Backgrounds & scales
    ax.Color  = bgGrey;          % plotting area bg
    ax.YScale = 'log';
    xlabel(ax,'Time (h)');
    yl = ylabel(ax,'No. of bacterial cells (Bacteria/mL)','Interpreter','none');
    title(ax,'Bacteria Growth Curves');

    % Default fixed Y-limits for log mode
    fixedLogYLim = [1e3 1e13];
    ylim(ax, fixedLogYLim);

    % Lines (created once, updated in callback)
    hExp  = plot(ax, t, nan(size(t)), 'LineWidth', 1.8, 'DisplayName','Exponential');
    hLog  = plot(ax, t, nan(size(t)), 'LineWidth', 1.8, 'DisplayName','Logistic');
    hGomp = plot(ax, t, nan(size(t)), 'LineWidth', 1.8, 'DisplayName','Gompertz (Zwietering)');

    % K reference line (styled, bold label)
    hKref = yline(ax, defaults.K, '--', 'K (carrying capacity)', ...
        'LabelHorizontalAlignment','left', 'Color', kColor, 'LineWidth', 2.4, ...
        'DisplayName','K (carrying capacity)');
    try
        hKref.Label.FontSize = 14;
        hKref.Label.FontWeight = 'bold';
    catch, end

    % Legend (explicit order)
    lgd = legend(ax, [hExp hLog hGomp hKref], ...
        'Location','southoutside','Orientation','horizontal');
    lgd.Box = 'off';

    %% UI controls panel
    p = uipanel('Parent',f,'Title','Parameters','FontWeight','bold', ...
                'Position',[0.72 0.18 0.26 0.75], ...
                'BackgroundColor', bgGrey);

    uicontrol(p,'Style','text','String','Model to display:',...
        'HorizontalAlignment','left','Position',[15 420 200 20], ...
        'BackgroundColor', bgGrey);
    modelMenu = uicontrol(p,'Style','popupmenu', ...
        'String',{'All','Exponential','Logistic','Gompertz'}, ...
        'Position',[15 400 230 22],'Callback',@refresh);

    % Y-axis scale selector
    uicontrol(p,'Style','text','String','Y-axis scale:',...
        'HorizontalAlignment','left','Position',[15 365 200 20], ...
        'BackgroundColor', bgGrey);
    yScaleMenu = uicontrol(p,'Style','popupmenu', ...
        'String',{'Logarithmic','Linear (to K/Max)'}, ...
        'Value',1,'Position',[15 345 230 22],'Callback',@refresh);

    % N0
    uicontrol(p,'Style','text','String','Initial population N0:',...
        'HorizontalAlignment','left','Position',[15 305 200 20], ...
        'BackgroundColor', bgGrey);
    N0Edit = uicontrol(p,'Style','edit','String',num2str(defaults.N0), ...
        'Position',[15 285 230 24],'Callback',@refresh);

    % ---- Carrying capacity K (log slider) with dynamic range ----
    loOrderDefault = floor(log10(defaults.N0)) - 1;
    hiOrderDefault = ceil(log10(defaults.K)) + 1;
    kMin = max(-3, loOrderDefault);
    kMax = min(18, hiOrderDefault);
    if kMax <= kMin, kMax = kMin + 1; end
    kVal0 = min(max(log10(defaults.K), kMin), kMax);

    uicontrol(p,'Style','text','String','Carrying capacity K (log slider):', ...
        'HorizontalAlignment','left','Position',[15 250 230 20], ...
        'BackgroundColor', bgGrey);
    KSlider = uicontrol(p,'Style','slider','Min',kMin,'Max',kMax,'Value',kVal0, ...
        'Position',[15 230 230 20],'Callback',@refresh);
    KValLab = uicontrol(p,'Style','text','String',sprintf('K = %s',eng(10.^kVal0)), ...
        'HorizontalAlignment','left','Position',[15 205 230 20], ...
        'BackgroundColor', bgGrey);

    % Growth rate r
    uicontrol(p,'Style','text','String','Growth rate r (1/h):', ...
        'HorizontalAlignment','left','Position',[15 175 230 20], ...
        'BackgroundColor', bgGrey);
    rSlider = uicontrol(p,'Style','slider','Min',0.01,'Max',3.0,'Value',defaults.r, ...
        'Position',[15 155 230 20],'Callback',@refresh);
    rValLab = uicontrol(p,'Style','text','String',sprintf('r = %.3f 1/h',defaults.r), ...
        'HorizontalAlignment','left','Position',[15 130 230 20], ...
        'BackgroundColor', bgGrey);

    % Lag time lambda (Gompertz only)
    uicontrol(p,'Style','text','String','Lag time λ (h, Gompertz only):', ...
        'HorizontalAlignment','left','Position',[15 100 230 20], ...
        'BackgroundColor', bgGrey);
    lamSlider = uicontrol(p,'Style','slider','Min',0,'Max',24,'Value',defaults.lam, ...
        'Position',[15 80 230 20],'Callback',@refresh);
    lamValLab = uicontrol(p,'Style','text','String',sprintf('λ = %.2f h',defaults.lam), ...
        'HorizontalAlignment','left','Position',[15 55 230 20], ...
        'BackgroundColor', bgGrey);

    % Presets
    uicontrol(p,'Style','pushbutton','String','Preset: Slow growth', ...
        'Position',[15 25 110 24],'Callback',@(~,~)setPreset(1));
    uicontrol(p,'Style','pushbutton','String','Preset: Fast growth', ...
        'Position',[135 25 110 24],'Callback',@(~,~)setPreset(2));

    %% Lag visuals (state kept in outer scope, no persistent)
    lagPatch = gobjects(1);
    hLag     = gobjects(1);
    ylListener = [];

    % Initial render
    refresh();

    %% Nested callbacks and helpers
    function setPreset(which)
        % Keep N0 editable but reset to default for the preset
        N0Edit.String = num2str(defaults.N0);
        switch which
            case 1 % Slow
                KSlider.Value   = ensureSliderCanHold(KSlider, log10(1e7));
                rSlider.Value   = 0.30;
                lamSlider.Value = 6.00;
            case 2 % Fast
                KSlider.Value   = ensureSliderCanHold(KSlider, log10(1e9));
                rSlider.Value   = 1.20;
                lamSlider.Value = 2.00;
        end
        refresh();
    end

    function refresh(~,~)
        % --- Read parameters
        N0  = str2double(N0Edit.String);
        if ~isfinite(N0) || N0 <= 0, N0 = defaults.N0; N0Edit.String = num2str(N0); end
        K   = 10.^get(KSlider,'Value');
        r   = get(rSlider,'Value');
        lam = get(lamSlider,'Value');

        % --- Enforce K > N0 for Gompertz (softly bump if needed)
        if K <= N0
            K = max(N0*1.01, N0 + 1);  % minimal separation
            set(KSlider,'Value', ensureSliderCanHold(KSlider, log10(K)));
        end

        % --- Labels & reference
        KValLab.String   = sprintf('K = %s', eng(K));
        rValLab.String   = sprintf('r = %.3f 1/h', r);
        lamValLab.String = sprintf('λ = %.2f h', lam);
        hKref.Value      = K;

        % --- Compute curves
        Nexp  = model_exponential(t, N0, r);
        Nlog  = model_logistic(t, N0, K, r);
        Ngomp = model_gompertz_mod_zwietering(t, N0, K, r, lam);

        % --- Which models to show?
        choice = modelMenu.Value; % 1:All, 2:Exp, 3:Log, 4:Gomp
        vis = {'on','on','on'};
        switch choice
            case 2, vis = {'on','off','off'};
            case 3, vis = {'off','on','off'};
            case 4, vis = {'off','off','on'};
        end

        % --- Update plot data/visibility
        set(hExp , 'YData', Nexp , 'XData', t, 'Visible', vis{1});
        set(hLog , 'YData', Nlog , 'XData', t, 'Visible', vis{2});
        set(hGomp, 'YData', Ngomp, 'XData', t, 'Visible', vis{3});

        % --- Y-axis scale handling
        if yScaleMenu.Value == 1
            % Log (fixed)
            ax.YScale = 'log';
            ylim(ax, fixedLogYLim);
        else
            % Linear (auto, anchored to N0)
            ax.YScale = 'linear';
            switch choice
                case 1  % All: avoid exponential blow-up dominating if hidden
                    yStack = [Nlog(:); Ngomp(:); K]; % ignore Exp to keep view sane
                case 2  % Exponential only
                    yStack = [Nexp(:); K];
                case 3
                    yStack = [Nlog(:); K];
                case 4
                    yStack = [Ngomp(:); K];
            end
            yStack = yStack(isfinite(yStack) & yStack > 0);
            if isempty(yStack), yStack = max(N0, 1); end

            lowerPad = 0.10;   % 10% below N0
            upperPad = 0.10;   % 10% headroom above max

            ymin = max(eps, N0 * (1 - lowerPad));
            ymax = max(yStack);
            if ~(isfinite(ymax) && ymax > 0)
                ymax = N0 * 2;
            else
                ymax = ymax * (1 + upperPad);
            end
            if ymax <= ymin, ymax = ymin * 1.5; end
            ylim(ax, [ymin, ymax]);
        end

        % --- Lag visuals (shaded 0..λ, vertical λ line)
        yl_now = ylim(ax);
        if ~isgraphics(lagPatch)
            lagPatch = patch(ax, [0 lam lam 0], [yl_now(1) yl_now(1) yl_now(2) yl_now(2)], shadeCol, ...
                'FaceAlpha', 0.10, 'EdgeColor','none', 'HitTest','off', 'PickableParts','none');
            try
                lagPatch.Annotation.LegendInformation.IconDisplayStyle = 'off';
            catch, end
            uistack(lagPatch,'bottom');
        else
            set(lagPatch, 'XData', [0 lam lam 0], 'YData', [yl_now(1) yl_now(1) yl_now(2) yl_now(2)]);
            uistack(lagPatch,'bottom');
        end

        if ~isgraphics(hLag)
            hLag = xline(ax, lam, ':', 'λ (lag time)', 'Color', lagColor, 'LineWidth', 2.4);
            try
                set(hLag, 'LabelOrientation','horizontal', ...
                          'LabelHorizontalAlignment','left', ...
                          'LabelVerticalAlignment','top');
                hLag.Label.FontSize = 14;
                hLag.Label.FontWeight = 'bold';
                hLag.Annotation.LegendInformation.IconDisplayStyle = 'off';
            catch
                % Older MATLAB fallbacks
                try
                    set(hLag, 'LabelOrientation','horizontal', ...
                              'LabelHorizontalAlignment','left');
                catch, end
                try
                    set(get(get(hLag,'Annotation'),'LegendInformation'),'IconDisplayStyle','off');
                catch, end
            end
        else
            hLag.Value = lam;
        end

        % Keep lag patch synced with Y-limits (install listener once)
        if isempty(ylListener) || ~isvalid(ylListener)
            ylListener = addlistener(ax, 'YLim', 'PostSet', @(~,~) syncLagPatch());
        end
        syncLagPatch();

        drawnow;
    end

    %% Helper: keep lag patch spanning current y-limits
    function syncLagPatch()
        if isgraphics(lagPatch)
            yl_now = ylim(ax);
            xdat = get(lagPatch,'XData');
            set(lagPatch,'YData',[yl_now(1) yl_now(1) yl_now(2) yl_now(2)], 'XData', xdat);
            uistack(lagPatch,'bottom');
        end
    end

    %% Helper: safely set slider value, expanding its range if needed
    function val = ensureSliderCanHold(sl, val)
        if val < sl.Min
            try, sl.Min = floor(val); catch, end
        end
        if val > sl.Max
            try, sl.Max = ceil(val); catch, end
        end
        val = min(max(val, sl.Min), sl.Max);
    end

    %% Helper: positive finite
    function x = validpos(x)
        x(~isfinite(x) | x<=0) = eps;
    end

    %% Helper: engineering notation (compact K label)
    function s = eng(x)
        if ~isfinite(x) || x==0, s = '0'; return; end
        e3 = floor(log10(abs(x))/3)*3;
        e3 = max(-24, min(24, e3)); % clamp
        mant = x / 10^e3;
        sfx  = containers.Map( ...
            num2cell(-24:3:24), ...
            {'y','z','a','f','p','n','µ','m','','k','M','G','T','P','E','Z','Y'} );
        if sfx.isKey(e3)
            suf = sfx(e3);
        else
            suf = sprintf('e%d',e3);
        end
        s = sprintf('%.3g%s', mant, suf);
    end

    %% Helper: check font availability
    function tf = localFontAvailable(fname)
        tf = false;
        try
            list = listfonts; % may be slow on some systems but acceptable here
            tf = any(strcmpi(list, fname));
        catch
            tf = false;
        end
    end

    %% Models
    function N = model_exponential(t, N0, r)
        % Pure exponential (no lag)
        N = N0 .* exp(r .* t);
        N = validpos(N);
    end

    function N = model_logistic(t, N0, K, r)
        % Classic logistic (no lag)
        N = K ./ ( 1 + ((K - N0)./N0) .* exp(-r .* t) );
        N = validpos(N);
    end

    function N = model_gompertz_mod_zwietering(t, N0, K, r, lam)
        % Modified Gompertz (Zwietering 1990)
        % A = ln(K/N0); r = mu_max; λ = lag
        A = log(K./N0);
        A = max(A, 1e-9); % stability
        N = N0 .* exp( A .* exp( -exp( (r*exp(1)./A).*(lam - t) + 1 ) ) );
        N = validpos(N);
    end
end